// background.js - 后台服务工作者

// 存储当前下载信息
let currentDownloadInfo = null;

// 监听来自内容脚本的消息
chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
  console.log('收到消息:', request);
  
  if (request.action === 'setDownloadInfo') {
    // 存储下载信息
    currentDownloadInfo = {
      downloadUrl: request.downloadUrl,
      cookieName: request.cookieName,
      cookieValue: request.cookieValue,
      cookieDomain: request.cookieDomain,
      fileName: request.fileName
    };
    
    chrome.storage.local.set({
      downloadInfo: currentDownloadInfo
    }, function() {
      console.log('下载信息已存储');
      sendResponse({ success: true });
    });
    
    // 返回true表示异步响应
    return true;
  }
  
  if (request.action === 'setDownloadCookie') {
    // 设置下载Cookie
    chrome.cookies.set({
      url: `https://${request.cookieDomain}`,
      name: request.cookieName,
      value: request.cookieValue,
      domain: request.cookieDomain,
      path: '/',
      httpOnly: true,
      secure: true,
      sameSite: 'no_restriction'
    }, function(cookie) {
      if (chrome.runtime.lastError) {
        console.error('设置Cookie失败:', chrome.runtime.lastError);
        sendResponse({ success: false, error: chrome.runtime.lastError.message });
      } else {
        console.log('Cookie设置成功:', cookie);
        sendResponse({ success: true });
      }
    });
    
    // 返回true表示异步响应
    return true;
  }
  
  if (request.action === 'requestDownload') {
    // 处理下载请求
    handleDownloadRequest(request.fileid)
      .then(() => {
        sendResponse({ success: true });
      })
      .catch(error => {
        console.error('处理下载请求失败:', error);
        sendResponse({ success: false, error: error.message });
      });
    
    // 返回true表示异步响应
    return true;
  }
  
  if (request.action === 'fetchDownloadInfo') {
    // 获取下载信息
    fetchDownloadInfo(request.fileid)
      .then(data => {
        sendResponse({ success: true, data: data });
      })
      .catch(error => {
        sendResponse({ success: false, error: error.message });
      });
    
    // 返回true表示异步响应
    return true;
  }
  
  if (request.action === 'getCurrentDownloadInfo') {
    // 获取当前下载信息
    sendResponse({ success: true, data: currentDownloadInfo });
    return true;
  }
});

// 处理下载请求
async function handleDownloadRequest(fileid) {
  try {
    console.log('开始处理下载请求，文件ID:', fileid);
    
    // 直接获取下载信息
    const downloadInfo = await fetchDownloadInfo(fileid);
    
    // 存储下载信息
    currentDownloadInfo = downloadInfo;
    
    await chrome.storage.local.set({
      downloadInfo: currentDownloadInfo
    });
    
    console.log('下载信息已获取并存储:', currentDownloadInfo);
    
    // 打开扩展弹窗
    chrome.action.openPopup();
  } catch (error) {
    console.error('处理下载请求失败:', error);
    throw error;
  }
}

// 获取下载信息
async function fetchDownloadInfo(fileid) {
  try {
    console.log('开始调用API获取下载信息，文件ID:', fileid);
    
    // 使用 fetch API 来避免 CORS 问题
    const response = await fetch(`http://localhost/api/get_download_info.php?fileid=${fileid}`, {
      method: 'GET',
      credentials: 'include' // 包含凭证
    });
    
    if (!response.ok) {
      throw new Error(`HTTP错误: ${response.status}`);
    }
    
    const text = await response.text();
    const result = JSON.parse(text);
    console.log('API响应数据:', result);
    
    if (result.errcode === 0) {
      // 设置下载所需的 Cookie
      const downloadInfo = result.data;
      await chrome.cookies.set({
        url: `https://${downloadInfo.cookie_domain}`,
        name: downloadInfo.cookie_name,
        value: downloadInfo.cookie_value,
        domain: downloadInfo.cookie_domain,
        path: '/',
        httpOnly: true,
        secure: true,
        sameSite: 'no_restriction'
      });
      
      return downloadInfo;
    } else {
      throw new Error(result.errmsg || '获取下载信息失败');
    }
  } catch (error) {
    console.error('获取下载信息时发生错误:', error);
    throw error;
  }
}

// 当扩展安装时
chrome.runtime.onInstalled.addListener(function() {
  console.log('企业微信文件下载助手扩展已安装');
});