// content.js - 内容脚本

console.log('内容脚本已加载');

// 注入脚本到页面
const script = document.createElement('script');
script.src = chrome.runtime.getURL('injected.js');
(document.head || document.documentElement).appendChild(script);
script.onload = function() {
  console.log('注入脚本已加载');
  script.remove();
};

// Base58解码函数
function base58Decode(data) {
  const alphabet = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz';
  const base = alphabet.length;
  
  console.log('Base58解码开始，输入长度:', data.length);
  
  // 将Base58字符串转换为字节数组
  let result = [0];
  
  for (let i = 0; i < data.length; i++) {
    let carry = alphabet.indexOf(data[i]);
    if (carry === -1) {
      throw new Error('Invalid base58 character at position ' + i + ': ' + data[i]);
    }
    
    // 乘以基数并加上当前字符值
    for (let j = 0; j < result.length; j++) {
      carry += result[j] * base;
      result[j] = carry & 0xff;
      carry >>= 8;
    }
    
    // 处理剩余的进位
    while (carry > 0) {
      result.push(carry & 0xff);
      carry >>= 8;
    }
  }
  
  // 处理前导零（字符'1'）
  let leadingZeros = 0;
  for (let i = 0; i < data.length && data[i] === '1'; i++) {
    leadingZeros++;
  }
  
  // 添加前导零
  let bytes = [];
  for (let i = 0; i < leadingZeros; i++) {
    bytes.push(0);
  }
  
  // 反转结果并添加到字节数组
  result.reverse();
  for (let i = 0; i < result.length; i++) {
    bytes.push(result[i]);
  }
  
  console.log('Base58解码完成，输出字节数:', bytes.length);
  return bytes;
}

// 将字节数组转换为字符串
function bytesToString(bytes) {
  return bytes.map(byte => String.fromCharCode(byte)).join('');
}

// 解析加密的下载信息
function parseEncryptedInfo(encryptedInfo) {
  try {
    console.log('开始解析加密信息，长度:', encryptedInfo.length);
    console.log('加密信息前50字符:', encryptedInfo.substring(0, 50) + '...');
    
    if (!encryptedInfo || encryptedInfo.length === 0) {
      throw new Error('加密信息为空');
    }
    
    const decodedBytes = base58Decode(encryptedInfo);
    console.log('Base58解码成功，字节数:', decodedBytes.length);
    
    const jsonString = bytesToString(decodedBytes);
    console.log('JSON字符串:', jsonString.substring(0, 200) + '...');
    
    const downloadInfo = JSON.parse(jsonString);
    console.log('JSON解析成功，下载信息:', downloadInfo);
    
    // 验证必要字段
    if (!downloadInfo.download_url || !downloadInfo.file_name) {
      throw new Error('下载信息不完整，缺少必要字段');
    }
    
    return downloadInfo;
  } catch (error) {
    console.error('解密下载信息失败:', error);
    console.error('原始加密信息:', encryptedInfo);
    throw new Error('解密下载信息失败: ' + error.message);
  }
}

// 显示错误对话框
function showErrorDialog(fileid, errorMessage) {
    removeDialog();
    
    const dialog = document.createElement('div');
    dialog.id = 'download-dialog';
    dialog.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 10000;
        display: flex;
        justify-content: center;
        align-items: center;
        font-family: Arial, sans-serif;
    `;
    
    dialog.innerHTML = `
        <div style="background-color: white; padding: 20px; border-radius: 10px; max-width: 500px; text-align: center;">
            <h3 style="color: #d32f2f; margin-bottom: 15px;">下载错误</h3>
            <p style="color: #666; margin-bottom: 20px; line-height: 1.5;">
                文件ID: <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px;">${fileid}</code><br>
                错误信息: ${errorMessage}
            </p>
            <div style="display: flex; gap: 10px; justify-content: center;">
                <button id="retry-download" style="
                    background-color: #1976d2;
                    color: white;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 5px;
                    cursor: pointer;
                ">重试</button>
                <button id="close-error" style="
                    background-color: #f5f5f5;
                    color: #333;
                    border: 1px solid #ddd;
                    padding: 10px 20px;
                    border-radius: 5px;
                    cursor: pointer;
                ">关闭</button>
            </div>
        </div>
    `;
    
    document.body.appendChild(dialog);
    
    // 重试按钮事件
    dialog.querySelector('#retry-download').addEventListener('click', () => {
        dialog.remove();
        // 重新获取下载信息（不使用缓存）
        showDownloadDialog(fileid);
    });
    
    // 关闭按钮事件
    dialog.querySelector('#close-error').addEventListener('click', () => {
        dialog.remove();
    });
}

// 页面加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
  console.log('DOMContentLoaded 事件触发');
  
  // 监听来自popup或background的消息
  chrome.runtime.onMessage.addListener(function(request, sender, sendResponse) {
    console.log('收到扩展消息:', request);
    
    if (request.action === 'getDownloadInfo') {
      console.log('开始获取下载信息，文件ID:', request.fileid);
      
      // 通过 background script 获取下载信息，避免 CORS 问题
      chrome.runtime.sendMessage({
        action: 'fetchDownloadInfo',
        fileid: request.fileid
      }, function(response) {
        if (response.success) {
          console.log('成功获取下载信息:', response.data);
          sendResponse({ success: true, data: response.data });
        } else {
          console.error('获取下载信息失败:', response.error);
          sendResponse({ success: false, error: response.error });
        }
      });
      
      // 返回true表示异步响应
      return true;
    }
  });
  
  // 监听页面中的下载按钮点击事件
  document.addEventListener('click', function(event) {
    // 检查是否点击了下载按钮 (根据实际页面结构调整选择器)
    if (event.target.matches('.download-btn') || 
        event.target.closest('.download-btn')) {
      
      console.log('检测到下载按钮点击事件');
      
      // 阻止默认行为
      event.preventDefault();
      
      // 获取下载按钮元素
      const downloadBtn = event.target.closest('.download-btn');
      
      // 从 data-fileid 属性获取文件ID
      let fileid = '';
      if (downloadBtn) {
        fileid = downloadBtn.dataset.fileid || downloadBtn.getAttribute('data-fileid');
      }
      
      console.log('提取到文件ID:', fileid);
      
      if (fileid) {
        // 检测扩展是否安装
        checkExtensionInstalled(function(installed) {
          if (installed) {
            handleDownloadButtonClick(fileid, downloadBtn);
          } else {
            alert('请先安装浏览器扩展以使用下载功能');
          }
        });
      } else {
        console.error('无法获取文件ID，下载按钮元素:', downloadBtn);
        alert('无法获取文件ID，请刷新页面后重试');
      }
    }
  });
  
  // 监听来自注入脚本的消息
  window.addEventListener('message', function(event) {
    // 处理扩展安装状态响应
    if (event.data && event.data.type === 'EXTENSION_INSTALLED_RESPONSE') {
      window.extensionInstalled = event.data.installed;
    }
    
    if (event.data && event.data.type === 'DOWNLOAD_REQUEST_FROM_PAGE') {
      const fileid = event.data.fileid;
      if (fileid) {
        // 发送消息到扩展，请求获取下载信息
        handleDownloadButtonClick(fileid);
      }
    }
  });
});

// 检测扩展是否已安装
function checkExtensionInstalled(callback) {
  if (typeof window.extensionInstalled !== 'undefined') {
    callback(window.extensionInstalled);
    return;
  }
  
  // 发送消息到页面检查扩展
  window.postMessage({
    type: 'CHECK_EXTENSION_INSTALLED'
  }, '*');
  
  // 等待响应
  let timeout = setTimeout(function() {
    callback(false);
  }, 1000);
  
  function checkResponse(event) {
    if (event.data && event.data.type === 'EXTENSION_INSTALLED_RESPONSE') {
      clearTimeout(timeout);
      window.removeEventListener('message', checkResponse);
      callback(event.data.installed);
    }
  }
  
  window.addEventListener('message', checkResponse);
}

// 修改下载按钮点击处理
function handleDownloadButtonClick(fileid, buttonElement) {
    console.log('handleDownloadButtonClick 被调用');
    console.log('文件ID:', fileid);
    console.log('按钮元素:', buttonElement);
    
    // 尝试从按钮的data-encrypted-info属性获取加密信息
    const encryptedInfo = buttonElement ? buttonElement.dataset.encryptedInfo : null;
    
    console.log('加密信息存在性:', !!encryptedInfo);
    if (encryptedInfo) {
      console.log('加密信息长度:', encryptedInfo.length);
      console.log('加密信息前50字符:', encryptedInfo.substring(0, 50) + '...');
    }
    
    if (encryptedInfo && encryptedInfo.length > 0) {
      try {
        console.log('开始解析加密信息...');
        // 解析加密信息
        const downloadInfo = parseEncryptedInfo(encryptedInfo);
        console.log('加密信息解析成功，显示下载对话框');
        showDownloadDialog(fileid, downloadInfo);
      } catch (error) {
        console.error('解析加密信息失败:', error);
        console.error('错误详情:', error.message, error.stack);
        
        // 显示友好的错误信息
        const errorMessage = '解析下载信息失败: ' + error.message;
        console.error(errorMessage);
        
        // 显示错误对话框而不是alert
        showErrorDialog(fileid, errorMessage);
      }
    } else {
      console.log('没有加密信息，使用API方式获取下载信息');
      // 如果没有加密信息，则使用原来的API方式获取
      showDownloadDialog(fileid);
    }
}

// 显示下载对话框
function showDownloadDialog(fileid, downloadInfo = null) {
    removeDialog();
    
    const dialog = document.createElement('div');
    dialog.id = 'download-dialog';
    dialog.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        z-index: 9999;
        display: flex;
        justify-content: center;
        align-items: center;
        font-family: Arial, sans-serif;
    `;
    
    if (downloadInfo) {
        dialog.innerHTML = `
            <div style="background-color: white; padding: 20px; border-radius: 10px; max-width: 400px; text-align: center;">
                <h3 style="color: #4CAF50; margin-bottom: 15px;">准备下载</h3>
                <p style="color: #666; margin-bottom: 20px;">
                    文件名: <strong>${downloadInfo.file_name}</strong><br>
                    文件ID: <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px; font-size: 12px;">${fileid}</code>
                </p>
                <button id="confirm-download" style="
                    background-color: #4CAF50;
                    color: white;
                    border: none;
                    padding: 12px 30px;
                    border-radius: 5px;
                    cursor: pointer;
                    font-size: 16px;
                ">开始下载</button>
            </div>
        `;
        
        // 确认下载按钮事件
        dialog.querySelector('#confirm-download').addEventListener('click', () => {
            startDownload(downloadInfo);
            dialog.remove();
        });
    } else {
        dialog.innerHTML = `
            <div style="background-color: white; padding: 20px; border-radius: 10px; max-width: 400px; text-align: center;">
                <h3 style="color: #FF9800; margin-bottom: 15px;">获取下载信息</h3>
                <p style="color: #666; margin-bottom: 20px;">
                    正在获取文件下载信息...<br>
                    文件ID: <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px; font-size: 12px;">${fileid}</code>
                </p>
                <div style="color: #999; font-size: 14px;">请稍候...</div>
            </div>
        `;
        
        // 通过API获取下载信息
        chrome.runtime.sendMessage({
            action: 'fetchDownloadInfo',
            fileid: fileid
        }, function(response) {
            if (response && response.success) {
                dialog.remove();
                showDownloadDialog(fileid, response.data);
            } else {
                dialog.innerHTML = `
                    <div style="background-color: white; padding: 20px; border-radius: 10px; max-width: 400px; text-align: center;">
                        <h3 style="color: #F44336; margin-bottom: 15px;">获取失败</h3>
                        <p style="color: #666; margin-bottom: 20px;">
                            无法获取下载信息<br>
                            错误: ${response ? response.error : '未知错误'}
                        </p>
                        <button id="close-dialog" style="
                            background-color: #f5f5f5;
                            color: #333;
                            border: 1px solid #ddd;
                            padding: 10px 20px;
                            border-radius: 5px;
                            cursor: pointer;
                        ">关闭</button>
                    </div>
                `;
                
                dialog.querySelector('#close-dialog').addEventListener('click', () => {
                    dialog.remove();
                });
            }
        });
    }
    
    document.body.appendChild(dialog);
}

// 开始下载
function startDownload(downloadInfo) {
    console.log('开始下载:', downloadInfo);
    
    // 创建下载链接
    const link = document.createElement('a');
    link.href = downloadInfo.download_url;
    link.download = downloadInfo.file_name;
    
    // 设置cookie
    document.cookie = `${downloadInfo.cookie_name}=${downloadInfo.cookie_value}; path=/; domain=${downloadInfo.cookie_domain || ''}`;
    
    // 触发下载
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// 移除对话框
function removeDialog() {
    const dialog = document.getElementById('download-dialog');
    if (dialog) {
        dialog.remove();
    }
}